/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package chart;

import java.awt.Color;

import rdc.RDCSample;

import com.inet.report.Chart2;
import com.inet.report.DatabaseField;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.RDC;
import com.inet.report.ReportException;
import com.inet.report.Section;
import com.inet.report.SummaryField;
import com.inet.report.TableSource;
import com.inet.report.chart.ChartTitle;
import com.inet.report.chart.axis.ContinuousNumberAxis;
import com.inet.report.chart.axis.GroupAxis;
import com.inet.report.chart.dataset.TwoGroupsDataset;
import com.inet.report.chart.format.Abbreviation;
import com.inet.report.chart.format.FixedNumberFormat;
import com.inet.report.chart.plot.LinePlot;
import com.inet.report.chart.plot.LineStyle;

/**
 * This sample demonstrates how you can create a line chart and use this properties.
 */
public class LineSample extends RDCSample {

    /**
     * The category column name.
     */
    private static String category = "Category";

    /**
     * The series column name.
     */
    private static String series   = "Series";

    /**
     * The value column name.
     */
    private static String value    = "Value";

    /**
     * Creates an empty engine, adds a chart and sets the data.
     * @param exportFmt the export format set by super class.
     * @return returns the created engine.
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //Creates a new empty engine.
            Engine engine = RDC.createEmptyEngine( exportFmt );
            //Creates three database columns used by chart.            
            TableSource tableSource = engine.getDatabaseTables().getDatasource( 0 ).createTableSource( "sample" );
            tableSource.addColumn( category, Field.STRING );
            tableSource.addColumn( series, Field.STRING );
            tableSource.addColumn( value, Field.NUMBER );

            //Gets the first section of report header.            
            Section section = engine.getArea( Engine.REPORT_HEADER ).getSection( 0 );

            //Creates a new line chart with markers in the section.            
            Chart2 lineChart = section.addChart2( LineStyle.LINE_MARKER, 0, 0, 8000, 6000 );

            //Gets the header title of chart and sets the title label color.
            ChartTitle title = lineChart.getHeaderTitle();
            title.setColor( RDC.toCcColor( new Color( 85, 85, 85 ) ) );

            //Creates a data source of chart.
            LinePlot plot = (LinePlot)lineChart.getPlot();
            //This chart gets a two groups dataset.
            TwoGroupsDataset dataset = new TwoGroupsDataset( lineChart );
            //Adds the category field as group.
            DatabaseField categoryField = tableSource.getDatabaseField( category );
            dataset.setCategoryGroup( categoryField );
            //Adds the series field as group.
            DatabaseField seriesField = tableSource.getDatabaseField( series );
            dataset.setSeriesGroup( seriesField );
            //Adds the value field. The chart should shows a sum of values for each series in the category.
            DatabaseField valueField = tableSource.getDatabaseField( value );
            dataset.setDataField( SummaryField.SUM, valueField, null, 0 );
            plot.setDataset( dataset );

            //Gets the category axis to change it properties.
            GroupAxis categoryAxis = (GroupAxis)plot.getCategoryAxis();
            //Sets the axis grid lines visible.
            categoryAxis.setGridlinesVisible( true );

            //Gets the data axis to change it properties.
            ContinuousNumberAxis dataAxis = (ContinuousNumberAxis)plot.getDataAxis();
            //Sets a custom number format with thousand abbreviation and one decimal place.
            FixedNumberFormat format = new FixedNumberFormat( FixedNumberFormat.FORMAT_NUMBER );
            format.setAbbreviation( Abbreviation.ABBREVIATION_THOUSANDS );
            format.setNumberOfDecimalPlaces( 1 );
            dataAxis.setTickLabelFormat( format );

            //Sets the report data. 
            setData( engine );
            return engine;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Sets the report data. This sets the three data columns used by chart.
     * @param engine the current engine
     * @throws ReportException if engine is not initialized or finished.
     */
    private static void setData( Engine engine ) throws ReportException {
        String[] columns = { category, series, value };
        Object[][] data = new Object[6][];
        data[0] = new Object[3];
        data[0][0] = "Category 1";
        data[0][1] = "Series 1";
        data[0][2] = Integer.valueOf( 1000 );

        data[1] = new Object[3];
        data[1][0] = "Category 1";
        data[1][1] = "Series 2";
        data[1][2] = Integer.valueOf( 2000 );

        data[2] = new Object[3];
        data[2][0] = "Category 2";
        data[2][1] = "Series 1";
        data[2][2] = Integer.valueOf( 5000 );

        data[3] = new Object[3];
        data[3][0] = "Category 2";
        data[3][1] = "Series 2";
        data[3][2] = Integer.valueOf( 4000 );

        data[4] = new Object[3];
        data[4][0] = "Category 3";
        data[4][1] = "Series 1";
        data[4][2] = Integer.valueOf( 2000 );

        data[5] = new Object[3];
        data[5][0] = "Category 3";
        data[5][1] = "Series 2";
        data[5][2] = Integer.valueOf( 3000 );

        engine.setData( columns, data );
    }

    /**
     * Starts this sample.
     * @param argv nothing
     */
    public static void main( String[] argv ) {
        new LineSample().initUI();
    }
}
